%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Codes for Example 2.8
% Basic MCMC to estimate parameters of Lorenz-96 model 
% Created by John Harlim 
% Last edited: March 16, 2018  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear all, close all

CYC = 5000;                 % length of chain
N = 5;                      % dimension of state
T = 10;                     % length of model integration
dt = 0.05;                  % integration time step
u0 = sin([1:N]*2*pi/N)';    % initial condition
F = 8;                      % Lorenz-96 parameter
var = 0.01;                 % observation noise variance

% true state vector
u = zeros(N,T+1);            
u = l96rk4(1,F,u0,T);       

% observation
y = u + sqrt(var)*randn(N,T+1);

% MCMC initial condition
F_old = 9; 
SS_old = sum(diag((y-u)'*(y-u)));

%% MCMC parameter
R = sqrt(0.3);
a = 200;
b = a*var;
aval = a+.5*T;
bval = b + .5*SS_old;
sigma2 = 0.02;

%% MCMC simulation                  
for i = 1:CYC
    F_new = F_old + R*randn;
    u_new = l96rk4(1,F_new,u0,T);

    u_alpha = rand(1);
    SS_new = sum(diag((y-u_new)'*(y-u_new)));
    term = exp(-.5*(SS_new-SS_old)/sigma2);
    alpha = min(1,term);
    if u_alpha < alpha
      F_MCMC(i) = F_new;
      F_old = F_new;
      SS_old = SS_new;
    else
      F_MCMC(i) = F_old;
    end
    Sigma2(i) = sigma2;
    bval(i+1) = b + .5*SS_old;
    sigma2 = 1/gamrnd(aval,1/bval(i+1));
end

%% compute kde for the MCMC samples
range_F = max(F_MCMC) - min(F_MCMC);
F_min = min(F_MCMC)-range_F/10;
F_max = max(F_MCMC)+range_F/10;
[bandwidth_F,density_F,Fmesh,cdf_F]=kde(F_MCMC(1001:end));

range_S = max(Sigma2) - min(Sigma2);
S_min = min(Sigma2)-range_S/10;
S_max = max(Sigma2)+range_S/10;
[bandwidth_S,density_S,Smesh,cdf_S]=kde(Sigma2);


xx = [0:100:CYC];
nx = length(xx);
grey = [.4 .4 .4];

%% plot figure 2.2
figure(1)
subplot(2,1,1)
plot(F_MCMC,'--','color',grey,'linewidth',1)
hold on
plot(xx,F*ones(1,nx),'k','linewidth',2)
hold off
set(gca,'Fontsize',[16]);
axis([0 CYC 7 9])
ylabel('F')
legend('chain F_i','true value F')

subplot(2,1,2)
plot(Sigma2,'--','color',grey)
hold on
plot(xx,var*ones(1,nx),'k','linewidth',2)
hold off
xlabel('Iteration')
xlim([0 CYC])
set(gca,'Fontsize',[16]);
ylabel('\sigma^2')
legend('chain \sigma^2_i','true value \sigma^2')

%% plot figure 2.3
figure(2)
subplot(2,1,1)
plot(Fmesh,density_F,'k-','linewidth',2)
hold on
plot(mean(F_MCMC(1001:end)),0,'ko','markersize',8)
plot(F,0,'k*','markersize',8)
hold off
axis([7.5 8.5 0 10])
set(gca,'Fontsize',[16]);
xlabel('F')
legend('density estimate','mean estimate','true value')

subplot(2,1,2)
plot(Smesh,density_S,'k-','linewidth',2)
hold on
plot(mean(Sigma2),0,'ko','markersize',8)
plot(var,0,'k*','markersize',8)
hold off
ylim([0 600])
set(gca,'Fontsize',[16]);
xlabel('\sigma^2')
legend('density estimate','mean estimate','true value')


                  
                  
                  
                  
                  
                  
                  
                  
                  
                  


